##############################################################################
# ThanCad 0.0.8 "DoSomething": 2dimensional CAD with raster support for engineers.
# 
# Copyright (C)  November 13, 2004  by Thanasis Stamos
# URL:     http://thancad.sourceforge.net
# e-mail:  cyberthanasis@excite.com
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details (www.gnu.org/licenses/gpl.html).
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
##############################################################################

"""\
ThanCad 0.0.8 "DoSomething": 2dimensional CAD with raster support for engineers.

This module defines the (limited) colour range of a popular CAD system.
"""

from __future__ import print_function
import p_ggen


thanPartialCols = """\
  0    0   0   0  black
  1  255   0   0  red
  2  255 255   0  yellow
  3    0 255   0  green
  4    0 255 255  cyan
  5    0   0 255  blue
  6  255   0 255  magenta
  7  255 255 255  white
  8  128 128 128
  9  192 192 192
 10  255   1   1
 11  255 127 127
 12  165   0   0
 13  165  82  82
 14  127   0   0
 15  127  63  63
 16   76   0   0
 17   76  38  38
 18   38   0   0
 19   38  19  19
 20  255  63   0
 21  255 159 127
 22  165  41   0
 23  165 103  82
 24  127  31   0
 25  127  79  63
 26   76  19   0
 27   76  47  38
 28   39   9   0
 29   38  23  19
 30  255 127   0
 31  255 191 127
 32  165  85   0
 33  165 124  82
 34  127  63   0
 35  127  95  63
 36   76  38   0
 37   76  57  38
 38   38  19   0
 39   38  28  19
 40  255 191   0
 41  255 223 127
 42  165 124   0
 43  165 145  82
 44  127  95   0
 45  127 111  63
 46   76  57   0
 47   76  66  38
 48   38  28   0
 49   38  33  19
 50  255 255   1
 51  255 255 127
 52  165 165   0
 53  165 165  82
 54  127 127   0
 55  127 127  63
 56   76  76   0
 57   76  76  38
 58   38  38   0
 59   38  38  19
 60  191 255   0
 61  223 255 127
 62  124 165   0
 63  145 165  82
 64   95 127   0
 65  111 127  63
 66   57  76   0
 67   66  76  38
 68   28  38   0
 69   33  38  19
 70  127 255   0
 71  191 255 127
 72   82 165   0
 73  124 165  82
 74   63 127   0
 75   95 127  63
 76   38  76   0
 77   57  76  38
 78   19  38   0
 79   28  38  19
 80   63 255   0
 81  159 255 127
 82   41 165   0
 83  103 165  82
 84   31 127   0
 85   79 127  63
 86   19  76   0
 87   47  76  38
 88    9  38   0
 89   23  38  19
 90    1 255   1
 91  127 255 127
 92    0 165   0
 93   82 165  82
 94    0 127   0
 95   63 127  63
 96    0  76   0
 97   38  76  38
 98    0  38   0
 99   19  38  19
100    0 255  63
101  127 255 159
102    0 165  41
103   82 165 103
104    0 127  31
105   63 127  79
106    0  76  19
107   38  76  47
108    0  38   9
109   19  38  23
110    0 255 127
111  127 255 191
112    0 165  82
113   82 165 124
114    0 127  63
115   63 127  95
116    0  76  38
117   38  76  57
118    0  38  19
119   19  38  28
120    0 255 191
121  127 255 223
122    0 165 124
123   82 165 145
124    0 127  95
125   63 127 111
126    0  76  57
127   38  76  66
128    0  38  28
129   19  38  33
130    1 255 255
131  127 255 255
132    0 165 165
133   82 165 165
134    0 127 127
135   63 127 127
136    0  76  76
137   38  76  76
138    0  38  38
139   19  38  38
140    0 191 255
141  127 223 255
142    0 124 165
143   82 145 165
144    0  95 127
145   63 111 127
146    0  57  76
147   38  66  76
148    0  28  38
149   19  33  38
150    0 127 255
151  127 191 255
152    0  82 165
153   82 124 165
154    0  63 127
155   63  96 127
156    0  38  76
157   38  57  76
158    0  19  38
159   19  28  38
160    0  63 255
161  127 159 255
162    0  41 165
163   82 103 165
164    0  31 127
165   63  79 127
166    0  19  76
167   38  47  76
168    0   9  38
169   19  23  38
170    1   1 255
171  127 127 255
172    0   0 165
173   82  82 165
174    0   0 127
175   63  63 127
176    0   0  76
177   38  38  76
178    0   0  38
179   19  19  38
180   63   0 255
181  159 127 255
182   41   0 165
183  103  82 165
184   31   0 127
185   79  63 127
186   19   0  76
187   47  38  76
188    9   0  38
189   23  19  38
190  127   0 255
191  191 127 255
192   82   0 165
193  124  82 165
194   63   0 127
195   95  63 127
196   38   0  76
197   57  38  76
198   19   0  38
199   28  19  38
200  191   0 255
201  223 127 255
202  124   0 165
203  145  82 165
204   95   0 127
205  111  63 127
206   57   0  76
207   66  38  76
208   28   0  38
209   33  19  38
210  255   1 255
211  255 127 255
212  165   0 165
213  165  82 165
214  127   0 127
215  127  63 127
216   76   0  76
217   76  38  76
218   38   0  38
219   38  19  38
220  255   0 191
221  255 127 223
222  165   0 124
223  165  82 145
224  127   0  95
225  127  63 111
226   76   0  57
227   76  38  66
228   38   0  28
229   38  19  33
230  255   0 127
231  255 127 191
232  165   0  82
233  165  82 124
234  127   0  63
235  127  63  95
236   76   0  38
237   76  38  57
238   38   0  19
239   38  19  28
240  255   0  63
241  255 127 159
242  165   0  41
243  165  82 103
244  127   0  31
245  127  63  79
246   76   0  19
247   76  38  47
248   38   0   9
249   38  19  23
250   84  84  84
251  118 118 118
252  152 152 152
253  186 186 186
254  220 220 220
255  252 252 252"""


# $XConsortium: rgb.txt,v 10.41 94/02/20 18:39:36 rws Exp $
thanNamedCols = """\
255 250 250		snow
248 248 255		ghost white
248 248 255		GhostWhite
245 245 245		white smoke
245 245 245		WhiteSmoke
220 220 220		gainsboro
255 250 240		floral white
255 250 240		FloralWhite
253 245 230		old lace
253 245 230		OldLace
250 240 230		linen
250 235 215		antique white
250 235 215		AntiqueWhite
255 239 213		papaya whip
255 239 213		PapayaWhip
255 235 205		blanched almond
255 235 205		BlanchedAlmond
255 228 196		bisque
255 218 185		peach puff
255 218 185		PeachPuff
255 222 173		navajo white
255 222 173		NavajoWhite
255 228 181		moccasin
255 248 220		cornsilk
255 255 240		ivory
255 250 205		lemon chiffon
255 250 205		LemonChiffon
255 245 238		seashell
240 255 240		honeydew
245 255 250		mint cream
245 255 250		MintCream
240 255 255		azure
240 248 255		alice blue
240 248 255		AliceBlue
230 230 250		lavender
255 240 245		lavender blush
255 240 245		LavenderBlush
255 228 225		misty rose
255 228 225		MistyRose
255 255 255		white
  0   0   0		black
 47  79  79		dark slate gray
 47  79  79		DarkSlateGray
 47  79  79		dark slate grey
 47  79  79		DarkSlateGrey
105 105 105		dim gray
105 105 105		DimGray
105 105 105		dim grey
105 105 105		DimGrey
112 128 144		slate gray
112 128 144		SlateGray
112 128 144		slate grey
112 128 144		SlateGrey
119 136 153		light slate gray
119 136 153		LightSlateGray
119 136 153		light slate grey
119 136 153		LightSlateGrey
190 190 190		gray
190 190 190		grey
211 211 211		light grey
211 211 211		LightGrey
211 211 211		light gray
211 211 211		LightGray
 25  25 112		midnight blue
 25  25 112		MidnightBlue
  0   0 128		navy
  0   0 128		navy blue
  0   0 128		NavyBlue
100 149 237		cornflower blue
100 149 237		CornflowerBlue
 72  61 139		dark slate blue
 72  61 139		DarkSlateBlue
106  90 205		slate blue
106  90 205		SlateBlue
123 104 238		medium slate blue
123 104 238		MediumSlateBlue
132 112 255		light slate blue
132 112 255		LightSlateBlue
  0   0 205		medium blue
  0   0 205		MediumBlue
 65 105 225		royal blue
 65 105 225		RoyalBlue
  0   0 255		blue
 30 144 255		dodger blue
 30 144 255		DodgerBlue
  0 191 255		deep sky blue
  0 191 255		DeepSkyBlue
135 206 235		sky blue
135 206 235		SkyBlue
135 206 250		light sky blue
135 206 250		LightSkyBlue
 70 130 180		steel blue
 70 130 180		SteelBlue
176 196 222		light steel blue
176 196 222		LightSteelBlue
173 216 230		light blue
173 216 230		LightBlue
176 224 230		powder blue
176 224 230		PowderBlue
175 238 238		pale turquoise
175 238 238		PaleTurquoise
  0 206 209		dark turquoise
  0 206 209		DarkTurquoise
 72 209 204		medium turquoise
 72 209 204		MediumTurquoise
 64 224 208		turquoise
  0 255 255		cyan
224 255 255		light cyan
224 255 255		LightCyan
 95 158 160		cadet blue
 95 158 160		CadetBlue
102 205 170		medium aquamarine
102 205 170		MediumAquamarine
127 255 212		aquamarine
  0 100   0		dark green
  0 100   0		DarkGreen
 85 107  47		dark olive green
 85 107  47		DarkOliveGreen
143 188 143		dark sea green
143 188 143		DarkSeaGreen
 46 139  87		sea green
 46 139  87		SeaGreen
 60 179 113		medium sea green
 60 179 113		MediumSeaGreen
 32 178 170		light sea green
 32 178 170		LightSeaGreen
152 251 152		pale green
152 251 152		PaleGreen
  0 255 127		spring green
  0 255 127		SpringGreen
124 252   0		lawn green
124 252   0		LawnGreen
  0 255   0		green
127 255   0		chartreuse
  0 250 154		medium spring green
  0 250 154		MediumSpringGreen
173 255  47		green yellow
173 255  47		GreenYellow
 50 205  50		lime green
 50 205  50		LimeGreen
154 205  50		yellow green
154 205  50		YellowGreen
 34 139  34		forest green
 34 139  34		ForestGreen
107 142  35		olive drab
107 142  35		OliveDrab
189 183 107		dark khaki
189 183 107		DarkKhaki
240 230 140		khaki
238 232 170		pale goldenrod
238 232 170		PaleGoldenrod
250 250 210		light goldenrod yellow
250 250 210		LightGoldenrodYellow
255 255 224		light yellow
255 255 224		LightYellow
255 255   0		yellow
255 215   0 		gold
238 221 130		light goldenrod
238 221 130		LightGoldenrod
218 165  32		goldenrod
184 134  11		dark goldenrod
184 134  11		DarkGoldenrod
188 143 143		rosy brown
188 143 143		RosyBrown
205  92  92		indian red
205  92  92		IndianRed
139  69  19		saddle brown
139  69  19		SaddleBrown
160  82  45		sienna
205 133  63		peru
222 184 135		burlywood
245 245 220		beige
245 222 179		wheat
244 164  96		sandy brown
244 164  96		SandyBrown
210 180 140		tan
210 105  30		chocolate
178  34  34		firebrick
165  42  42		brown
233 150 122		dark salmon
233 150 122		DarkSalmon
250 128 114		salmon
255 160 122		light salmon
255 160 122		LightSalmon
255 165   0		orange
255 140   0		dark orange
255 140   0		DarkOrange
255 127  80		coral
240 128 128		light coral
240 128 128		LightCoral
255  99  71		tomato
255  69   0		orange red
255  69   0		OrangeRed
255   0   0		red
255 105 180		hot pink
255 105 180		HotPink
255  20 147		deep pink
255  20 147		DeepPink
255 192 203		pink
255 182 193		light pink
255 182 193		LightPink
219 112 147		pale violet red
219 112 147		PaleVioletRed
176  48  96		maroon
199  21 133		medium violet red
199  21 133		MediumVioletRed
208  32 144		violet red
208  32 144		VioletRed
255   0 255		magenta
238 130 238		violet
221 160 221		plum
218 112 214		orchid
186  85 211		medium orchid
186  85 211		MediumOrchid
153  50 204		dark orchid
153  50 204		DarkOrchid
148   0 211		dark violet
148   0 211		DarkViolet
138  43 226		blue violet
138  43 226		BlueViolet
160  32 240		purple
147 112 219		medium purple
147 112 219		MediumPurple
216 191 216		thistle
255 250 250		snow1
238 233 233		snow2
205 201 201		snow3
139 137 137		snow4
255 245 238		seashell1
238 229 222		seashell2
205 197 191		seashell3
139 134 130		seashell4
255 239 219		AntiqueWhite1
238 223 204		AntiqueWhite2
205 192 176		AntiqueWhite3
139 131 120		AntiqueWhite4
255 228 196		bisque1
238 213 183		bisque2
205 183 158		bisque3
139 125 107		bisque4
255 218 185		PeachPuff1
238 203 173		PeachPuff2
205 175 149		PeachPuff3
139 119 101		PeachPuff4
255 222 173		NavajoWhite1
238 207 161		NavajoWhite2
205 179 139		NavajoWhite3
139 121	 94		NavajoWhite4
255 250 205		LemonChiffon1
238 233 191		LemonChiffon2
205 201 165		LemonChiffon3
139 137 112		LemonChiffon4
255 248 220		cornsilk1
238 232 205		cornsilk2
205 200 177		cornsilk3
139 136 120		cornsilk4
255 255 240		ivory1
238 238 224		ivory2
205 205 193		ivory3
139 139 131		ivory4
240 255 240		honeydew1
224 238 224		honeydew2
193 205 193		honeydew3
131 139 131		honeydew4
255 240 245		LavenderBlush1
238 224 229		LavenderBlush2
205 193 197		LavenderBlush3
139 131 134		LavenderBlush4
255 228 225		MistyRose1
238 213 210		MistyRose2
205 183 181		MistyRose3
139 125 123		MistyRose4
240 255 255		azure1
224 238 238		azure2
193 205 205		azure3
131 139 139		azure4
131 111 255		SlateBlue1
122 103 238		SlateBlue2
105  89 205		SlateBlue3
 71  60 139		SlateBlue4
 72 118 255		RoyalBlue1
 67 110 238		RoyalBlue2
 58  95 205		RoyalBlue3
 39  64 139		RoyalBlue4
  0   0 255		blue1
  0   0 238		blue2
  0   0 205		blue3
  0   0 139		blue4
 30 144 255		DodgerBlue1
 28 134 238		DodgerBlue2
 24 116 205		DodgerBlue3
 16  78 139		DodgerBlue4
 99 184 255		SteelBlue1
 92 172 238		SteelBlue2
 79 148 205		SteelBlue3
 54 100 139		SteelBlue4
  0 191 255		DeepSkyBlue1
  0 178 238		DeepSkyBlue2
  0 154 205		DeepSkyBlue3
  0 104 139		DeepSkyBlue4
135 206 255		SkyBlue1
126 192 238		SkyBlue2
108 166 205		SkyBlue3
 74 112 139		SkyBlue4
176 226 255		LightSkyBlue1
164 211 238		LightSkyBlue2
141 182 205		LightSkyBlue3
 96 123 139		LightSkyBlue4
198 226 255		SlateGray1
185 211 238		SlateGray2
159 182 205		SlateGray3
108 123 139		SlateGray4
202 225 255		LightSteelBlue1
188 210 238		LightSteelBlue2
162 181 205		LightSteelBlue3
110 123 139		LightSteelBlue4
191 239 255		LightBlue1
178 223 238		LightBlue2
154 192 205		LightBlue3
104 131 139		LightBlue4
224 255 255		LightCyan1
209 238 238		LightCyan2
180 205 205		LightCyan3
122 139 139		LightCyan4
187 255 255		PaleTurquoise1
174 238 238		PaleTurquoise2
150 205 205		PaleTurquoise3
102 139 139		PaleTurquoise4
152 245 255		CadetBlue1
142 229 238		CadetBlue2
122 197 205		CadetBlue3
 83 134 139		CadetBlue4
  0 245 255		turquoise1
  0 229 238		turquoise2
  0 197 205		turquoise3
  0 134 139		turquoise4
  0 255 255		cyan1
  0 238 238		cyan2
  0 205 205		cyan3
  0 139 139		cyan4
151 255 255		DarkSlateGray1
141 238 238		DarkSlateGray2
121 205 205		DarkSlateGray3
 82 139 139		DarkSlateGray4
127 255 212		aquamarine1
118 238 198		aquamarine2
102 205 170		aquamarine3
 69 139 116		aquamarine4
193 255 193		DarkSeaGreen1
180 238 180		DarkSeaGreen2
155 205 155		DarkSeaGreen3
105 139 105		DarkSeaGreen4
 84 255 159		SeaGreen1
 78 238 148		SeaGreen2
 67 205 128		SeaGreen3
 46 139	 87		SeaGreen4
154 255 154		PaleGreen1
144 238 144		PaleGreen2
124 205 124		PaleGreen3
 84 139	 84		PaleGreen4
  0 255 127		SpringGreen1
  0 238 118		SpringGreen2
  0 205 102		SpringGreen3
  0 139	 69		SpringGreen4
  0 255	  0		green1
  0 238	  0		green2
  0 205	  0		green3
  0 139	  0		green4
127 255	  0		chartreuse1
118 238	  0		chartreuse2
102 205	  0		chartreuse3
 69 139	  0		chartreuse4
192 255	 62		OliveDrab1
179 238	 58		OliveDrab2
154 205	 50		OliveDrab3
105 139	 34		OliveDrab4
202 255 112		DarkOliveGreen1
188 238 104		DarkOliveGreen2
162 205	 90		DarkOliveGreen3
110 139	 61		DarkOliveGreen4
255 246 143		khaki1
238 230 133		khaki2
205 198 115		khaki3
139 134	 78		khaki4
255 236 139		LightGoldenrod1
238 220 130		LightGoldenrod2
205 190 112		LightGoldenrod3
139 129	 76		LightGoldenrod4
255 255 224		LightYellow1
238 238 209		LightYellow2
205 205 180		LightYellow3
139 139 122		LightYellow4
255 255	  0		yellow1
238 238	  0		yellow2
205 205	  0		yellow3
139 139	  0		yellow4
255 215	  0		gold1
238 201	  0		gold2
205 173	  0		gold3
139 117	  0		gold4
255 193	 37		goldenrod1
238 180	 34		goldenrod2
205 155	 29		goldenrod3
139 105	 20		goldenrod4
255 185	 15		DarkGoldenrod1
238 173	 14		DarkGoldenrod2
205 149	 12		DarkGoldenrod3
139 101	  8		DarkGoldenrod4
255 193 193		RosyBrown1
238 180 180		RosyBrown2
205 155 155		RosyBrown3
139 105 105		RosyBrown4
255 106 106		IndianRed1
238  99	 99		IndianRed2
205  85	 85		IndianRed3
139  58	 58		IndianRed4
255 130	 71		sienna1
238 121	 66		sienna2
205 104	 57		sienna3
139  71	 38		sienna4
255 211 155		burlywood1
238 197 145		burlywood2
205 170 125		burlywood3
139 115	 85		burlywood4
255 231 186		wheat1
238 216 174		wheat2
205 186 150		wheat3
139 126 102		wheat4
255 165	 79		tan1
238 154	 73		tan2
205 133	 63		tan3
139  90	 43		tan4
255 127	 36		chocolate1
238 118	 33		chocolate2
205 102	 29		chocolate3
139  69	 19		chocolate4
255  48	 48		firebrick1
238  44	 44		firebrick2
205  38	 38		firebrick3
139  26	 26		firebrick4
255  64	 64		brown1
238  59	 59		brown2
205  51	 51		brown3
139  35	 35		brown4
255 140 105		salmon1
238 130	 98		salmon2
205 112	 84		salmon3
139  76	 57		salmon4
255 160 122		LightSalmon1
238 149 114		LightSalmon2
205 129	 98		LightSalmon3
139  87	 66		LightSalmon4
255 165	  0		orange1
238 154	  0		orange2
205 133	  0		orange3
139  90	  0		orange4
255 127	  0		DarkOrange1
238 118	  0		DarkOrange2
205 102	  0		DarkOrange3
139  69	  0		DarkOrange4
255 114	 86		coral1
238 106	 80		coral2
205  91	 69		coral3
139  62	 47		coral4
255  99	 71		tomato1
238  92	 66		tomato2
205  79	 57		tomato3
139  54	 38		tomato4
255  69	  0		OrangeRed1
238  64	  0		OrangeRed2
205  55	  0		OrangeRed3
139  37	  0		OrangeRed4
255   0	  0		red1
238   0	  0		red2
205   0	  0		red3
139   0	  0		red4
255  20 147		DeepPink1
238  18 137		DeepPink2
205  16 118		DeepPink3
139  10	 80		DeepPink4
255 110 180		HotPink1
238 106 167		HotPink2
205  96 144		HotPink3
139  58  98		HotPink4
255 181 197		pink1
238 169 184		pink2
205 145 158		pink3
139  99 108		pink4
255 174 185		LightPink1
238 162 173		LightPink2
205 140 149		LightPink3
139  95 101		LightPink4
255 130 171		PaleVioletRed1
238 121 159		PaleVioletRed2
205 104 137		PaleVioletRed3
139  71	 93		PaleVioletRed4
255  52 179		maroon1
238  48 167		maroon2
205  41 144		maroon3
139  28	 98		maroon4
255  62 150		VioletRed1
238  58 140		VioletRed2
205  50 120		VioletRed3
139  34	 82		VioletRed4
255   0 255		magenta1
238   0 238		magenta2
205   0 205		magenta3
139   0 139		magenta4
255 131 250		orchid1
238 122 233		orchid2
205 105 201		orchid3
139  71 137		orchid4
255 187 255		plum1
238 174 238		plum2
205 150 205		plum3
139 102 139		plum4
224 102 255		MediumOrchid1
209  95 238		MediumOrchid2
180  82 205		MediumOrchid3
122  55 139		MediumOrchid4
191  62 255		DarkOrchid1
178  58 238		DarkOrchid2
154  50 205		DarkOrchid3
104  34 139		DarkOrchid4
155  48 255		purple1
145  44 238		purple2
125  38 205		purple3
 85  26 139		purple4
171 130 255		MediumPurple1
159 121 238		MediumPurple2
137 104 205		MediumPurple3
 93  71 139		MediumPurple4
255 225 255		thistle1
238 210 238		thistle2
205 181 205		thistle3
139 123 139		thistle4
  0   0   0		gray0
  0   0   0		grey0
  3   3   3		gray1
  3   3   3		grey1
  5   5   5		gray2
  5   5   5		grey2
  8   8   8		gray3
  8   8   8		grey3
 10  10  10 		gray4
 10  10  10 		grey4
 13  13  13 		gray5
 13  13  13 		grey5
 15  15  15 		gray6
 15  15  15 		grey6
 18  18  18 		gray7
 18  18  18 		grey7
 20  20  20 		gray8
 20  20  20 		grey8
 23  23  23 		gray9
 23  23  23 		grey9
 26  26  26 		gray10
 26  26  26 		grey10
 28  28  28 		gray11
 28  28  28 		grey11
 31  31  31 		gray12
 31  31  31 		grey12
 33  33  33 		gray13
 33  33  33 		grey13
 36  36  36 		gray14
 36  36  36 		grey14
 38  38  38 		gray15
 38  38  38 		grey15
 41  41  41 		gray16
 41  41  41 		grey16
 43  43  43 		gray17
 43  43  43 		grey17
 46  46  46 		gray18
 46  46  46 		grey18
 48  48  48 		gray19
 48  48  48 		grey19
 51  51  51 		gray20
 51  51  51 		grey20
 54  54  54 		gray21
 54  54  54 		grey21
 56  56  56 		gray22
 56  56  56 		grey22
 59  59  59 		gray23
 59  59  59 		grey23
 61  61  61 		gray24
 61  61  61 		grey24
 64  64  64 		gray25
 64  64  64 		grey25
 66  66  66 		gray26
 66  66  66 		grey26
 69  69  69 		gray27
 69  69  69 		grey27
 71  71  71 		gray28
 71  71  71 		grey28
 74  74  74 		gray29
 74  74  74 		grey29
 77  77  77 		gray30
 77  77  77 		grey30
 79  79  79 		gray31
 79  79  79 		grey31
 82  82  82 		gray32
 82  82  82 		grey32
 84  84  84 		gray33
 84  84  84 		grey33
 87  87  87 		gray34
 87  87  87 		grey34
 89  89  89 		gray35
 89  89  89 		grey35
 92  92  92 		gray36
 92  92  92 		grey36
 94  94  94 		gray37
 94  94  94 		grey37
 97  97  97 		gray38
 97  97  97 		grey38
 99  99  99 		gray39
 99  99  99 		grey39
102 102 102 		gray40
102 102 102 		grey40
105 105 105 		gray41
105 105 105 		grey41
107 107 107 		gray42
107 107 107 		grey42
110 110 110 		gray43
110 110 110 		grey43
112 112 112 		gray44
112 112 112 		grey44
115 115 115 		gray45
115 115 115 		grey45
117 117 117 		gray46
117 117 117 		grey46
120 120 120 		gray47
120 120 120 		grey47
122 122 122 		gray48
122 122 122 		grey48
125 125 125 		gray49
125 125 125 		grey49
127 127 127 		gray50
127 127 127 		grey50
130 130 130 		gray51
130 130 130 		grey51
133 133 133 		gray52
133 133 133 		grey52
135 135 135 		gray53
135 135 135 		grey53
138 138 138 		gray54
138 138 138 		grey54
140 140 140 		gray55
140 140 140 		grey55
143 143 143 		gray56
143 143 143 		grey56
145 145 145 		gray57
145 145 145 		grey57
148 148 148 		gray58
148 148 148 		grey58
150 150 150 		gray59
150 150 150 		grey59
153 153 153 		gray60
153 153 153 		grey60
156 156 156 		gray61
156 156 156 		grey61
158 158 158 		gray62
158 158 158 		grey62
161 161 161 		gray63
161 161 161 		grey63
163 163 163 		gray64
163 163 163 		grey64
166 166 166 		gray65
166 166 166 		grey65
168 168 168 		gray66
168 168 168 		grey66
171 171 171 		gray67
171 171 171 		grey67
173 173 173 		gray68
173 173 173 		grey68
176 176 176 		gray69
176 176 176 		grey69
179 179 179 		gray70
179 179 179 		grey70
181 181 181 		gray71
181 181 181 		grey71
184 184 184 		gray72
184 184 184 		grey72
186 186 186 		gray73
186 186 186 		grey73
189 189 189 		gray74
189 189 189 		grey74
191 191 191 		gray75
191 191 191 		grey75
194 194 194 		gray76
194 194 194 		grey76
196 196 196 		gray77
196 196 196 		grey77
199 199 199 		gray78
199 199 199 		grey78
201 201 201 		gray79
201 201 201 		grey79
204 204 204 		gray80
204 204 204 		grey80
207 207 207 		gray81
207 207 207 		grey81
209 209 209 		gray82
209 209 209 		grey82
212 212 212 		gray83
212 212 212 		grey83
214 214 214 		gray84
214 214 214 		grey84
217 217 217 		gray85
217 217 217 		grey85
219 219 219 		gray86
219 219 219 		grey86
222 222 222 		gray87
222 222 222 		grey87
224 224 224 		gray88
224 224 224 		grey88
227 227 227 		gray89
227 227 227 		grey89
229 229 229 		gray90
229 229 229 		grey90
232 232 232 		gray91
232 232 232 		grey91
235 235 235 		gray92
235 235 235 		grey92
237 237 237 		gray93
237 237 237 		grey93
240 240 240 		gray94
240 240 240 		grey94
242 242 242 		gray95
242 242 242 		grey95
245 245 245 		gray96
245 245 245 		grey96
247 247 247 		gray97
247 247 247 		grey97
250 250 250 		gray98
250 250 250 		grey98
252 252 252 		gray99
252 252 252 		grey99
255 255 255 		gray100
255 255 255 		grey100
169 169 169		dark grey
169 169 169		DarkGrey
169 169 169		dark gray
169 169 169		DarkGray
0     0 139		dark blue
0     0 139		DarkBlue
0   139 139		dark cyan
0   139 139		DarkCyan
139   0 139		dark magenta
139   0 139		DarkMagenta
139   0   0		dark red
139   0   0		DarkRed
144 238 144		light green
144 238 144		LightGreen"""


def __tra():
    "Transforms the data to a dictionary."
    import sys
    from . import colrnames, xkcd_rgb
    cols = {}
    names = {}
    invnames = {}
    for dline in thanNamedCols.split("\n"):         #XConsortium named colours
        dl = dline.split(None, 3)
        col = tuple(map(int, dl[:3]))
        names[dl[3].lower()] = col
        invnames[col] = dl[3].lower()
    for dl, col in colrnames.THECOLORS.items(): #Python Game Library   #OK for python 2, 3
        names[dl.lower()] = col[:3]
        invnames[col[:3]] = dl.lower()
    for dl in xkcd_rgb.colors.split("\n"):
        dl, col = dl.strip().split("\t")
        dl = dl.lower()
        c1 = int(col[1:3], base=16)  # #9d0759
        c2 = int(col[3:5], base=16)
        c3 = int(col[5:7], base=16)
        col = c1, c2, c3
        if dl in names:
            #print("xkcd color name '{}' ('{}') already defined: {}".format(dl, col, names[dl]))
            continue
        elif col in invnames:
            #print("xkcd color tuple '{}' ('{}') already defined: {}".format(col, dl, invnames[col]))
            continue
        names[dl] = col
        invnames[col] = dl
    for dline in thanPartialCols.split("\n"):       #thAtCad dxf partial colours
        dl = dline.split()
        col = tuple(map(int, dl[1:4]))
        cols[int(dl[0])] = col
        try:
            names[dl[4]] = col
            invnames[col] = dl[4]
        except IndexError:
            pass
    try: del sys.modules["colrnames"], sys.modules["xkcd_rgb"]   # This is to assist cpy.py
    except: pass
    return cols, names, invnames

thanDxfColCode2Rgb = {}   #Not needed, but keeps eclipse happy
thanDxfColName2Rgb = {}
thanRgb2DxfColName = {}
thanRgb2DxfColCode = {}
thanDxfColCode2Rgb, thanDxfColName2Rgb, thanRgb2DxfColName = __tra()
thanRgb2DxfColCode = p_ggen.dictInvert(thanDxfColCode2Rgb)
thanFormTkcol = "#%02x%02x%02x"

#Dictionaries with self explanatory names; the above cryptic names 
#are kept for compatibility and they should be deleted.
# Dxf colour code: It is an integer 1-255 used in dxf files to represent a colour.
#                  These colours are a reasonable (partial) subset of the RGB colour space.
# Dxf colour name: It is a text which described the most basic colours.
# RGB colour     : It is a tuple of three integers 0-255 which correspond to Red, Green and Blue

del __tra, thanPartialCols, thanNamedCols


def thanRgb2DxfColCodeApprox(rgb):
    "Returns a dxf colour code that resembles the rgb colour."
    r,g,b = rgb; difmin = 1e30
    for partial, (r1,g1,b1) in thanDxfColCode2Rgb.items():   #OK for python 2, 3
        dif = abs(r-r1)*max(r,128) + \
              abs(g-g1)*max(g,128) + \
              abs(b-b1)*max(b,128)
        if dif < difmin: difmin = dif; partmin = partial
    return partmin


def thanRgb2DxfGrayCodeApprox(rgb):
    "Returns a dxf gray code that resembles the gray shade of the rgb colour."
    igray = thanRgb2Gray(rgb)
    return min( (abs(igray-r), cod) 
                for cod,(r,g,b) in thanDxfColCode2Rgb.items()   #OK for python 2, 3
                if r == g == b
              )[1]


def thanDxfColCode2DxfGrayCodeApprox(cod):
    "Returns a dxf gray code that the resembles gray shade of a dxf colour code."
    r,g,b = thanDxfColCode2Rgb[cod]
    igray = int(0.299*r + 0.587*g + 0.114*b + 0.5)
    return min( (abs(igray-r), cod) 
                for cod,(r,g,b) in thanDxfColCode2Rgb.items()  #OK for python 2, 3
                if r == g == b
              )[1]


def thanDxfColCode2Tk(cod):
    "Returns Tk representation of a dxf colour code."
    rgb = thanDxfColCode2Rgb[cod]
    return thanFormTkcol % rgb


def thanDxfColCommonCodes():
    "All dxf color codes which are not too dark."
    c = []
    for i in range(10):
        c.append(i)
    for i in range(10, 16):
        while i < 256:
            c.append(i)
            i += 10
    return c


def thanRgb2Gray(rgb):
    "Transform the colour to gray scale according to ITU-R 601-2 transform; return an integer."
    r, g, b = rgb
    igray = int(0.299*r + 0.587*g + 0.114*b + 0.5)
    assert 0 <= igray < 256, "Something wrong with the rgb value of the colour"
    return igray


def thanTk2Rgb(bcol):
    "Transforms Tk formatted color to RGB; may raise ValueError."
    bcol = str(bcol)
    try: return thanDxfColName2Rgb[bcol]
    except KeyError: pass
    r = int("0x"+bcol[1:3], 16)
    g = int("0x"+bcol[3:5], 16)
    b = int("0x"+bcol[5:7], 16)
    return r, g, b


def __prgray():
    "Find the gray colours in the dxf codes; for debugging only."
    print("gray dxf colours:")
    for partial, (r1,g1,b1) in thanDxfColCode2Rgb.items():  #OK for python 2, 3
        if r1==g1==b1:
            print(partial,":",r1,g1,b1)


if __name__ == "__main__": 
    print(__doc__)
    __prgray()
